// ****************************************************************************
// GlottalImageExplorer.
// Copyright (C) 2015-2016 Peter Birkholz.
// This program is free and open-source software.
// ****************************************************************************

#include "MainWindow.h"
#include "ImageProc.h"

#include <iostream>
#include <wx/statline.h>
#include <wx/busyinfo.h>

using namespace std;

// ****************************************************************************
// IDs.
// ****************************************************************************

// Menu
static const int IDM_LOAD_AVI_FILM = 1203;
static const int IDM_LOAD_RAW_FILM = 1204;
static const int IDM_LOAD_BMP_SEQUENCE      = 1205;
static const int IDM_LOAD_SEGMENTATION_DATA = 1206;
static const int IDM_SAVE_SEGMENTATION_DATA = 1207;
static const int IDM_EXPORT_GLOTTIS_AREA = 1208;
static const int IDM_EXPORT_GLOTTIS_CONTOUR = 1209;

// Side frame widgets
static const int IDR_IMAGE_TYPE_BOX         = 1300;
static const int IDC_SEED_THRESHOLD_COUPLING = 1301;
static const int IDC_SHOW_GLOTTIS_AREA = 1302;
static const int IDC_SHOW_GLOTTIS_EDGE = 1303;
static const int IDB_CLEAR_SEGMENTATION_DATA = 1305;
static const int IDB_CALC_GLOTTAL_AREA_CURVE = 1306;

// Main frame widgets
static const int IDS_FRAME_INDEX            = 1400;
static const int IDB_TO_START               = 1401;
static const int IDB_PLAY_ALL               = 1402;
static const int IDB_PLAY_SELECTION         = 1403;
static const int IDS_SIGNALS                = 1403;
static const int IDC_PLAYBACK_SPEED         = 1404;

// Accelerator keys
static const int IDK_CTRL_LEFT              = 1900;
static const int IDK_CTRL_RIGHT             = 1901;


// ****************************************************************************
// The event table.
// ****************************************************************************

BEGIN_EVENT_TABLE(MainWindow, wxFrame)
  EVT_CLOSE(OnCloseWindow)

  // Menu
  EVT_MENU(IDM_LOAD_AVI_FILM, OnLoadAviFilm)
  EVT_MENU(IDM_LOAD_RAW_FILM, OnLoadRawFilm)
  EVT_MENU(IDM_LOAD_BMP_SEQUENCE, OnLoadBmpSequence)
  EVT_MENU(IDM_LOAD_SEGMENTATION_DATA, OnLoadSegmentationData)
  EVT_MENU(IDM_SAVE_SEGMENTATION_DATA, OnSaveSegmentationData)
  EVT_MENU(IDM_EXPORT_GLOTTIS_AREA, OnExportGlottisArea)
  EVT_MENU(IDM_EXPORT_GLOTTIS_CONTOUR, OnExportGlottisContour)
  EVT_MENU(wxID_EXIT, OnExit)
  EVT_MENU(wxID_ABOUT, OnAbout)

  // Side frame widgets
  EVT_RADIOBOX(IDR_IMAGE_TYPE_BOX, OnSelectImageType)
  EVT_CHECKBOX(IDC_SEED_THRESHOLD_COUPLING, OnSeedThresholdCoupling)
  EVT_CHECKBOX(IDC_SHOW_GLOTTIS_AREA, OnShowGlottisArea)
  EVT_CHECKBOX(IDC_SHOW_GLOTTIS_EDGE, OnShowGlottisEdge)
  EVT_BUTTON(IDB_CLEAR_SEGMENTATION_DATA, OnClearSegmentationData)
  EVT_BUTTON(IDB_CALC_GLOTTAL_AREA_CURVE, OnCalcGlottalAreaCurve)

  // Main frame widgets
  EVT_MENU(PLAYBACK_THREAD_EVENT, OnPlaybackThreadEvent)
  EVT_BUTTON(IDB_TO_START, OnToStart)
  EVT_BUTTON(IDB_PLAY_ALL, OnStartPlayback)
  EVT_BUTTON(IDB_PLAY_SELECTION, OnStartSelectionPlayback)
  EVT_SPINCTRL(IDC_PLAYBACK_SPEED, OnPlaybackSpeedChanged)
  EVT_COMMAND_SCROLL(IDS_FRAME_INDEX, OnFrameIndexScroll)
  EVT_COMMAND_SCROLL(IDS_SIGNALS, OnSignalPictureScroll)

  // Accelerator keys must be handled with a menu event
  EVT_MENU(IDK_CTRL_LEFT, OnKeyCtrlLeft)
  EVT_MENU(IDK_CTRL_RIGHT, OnKeyCtrlRight)
END_EVENT_TABLE()


// ****************************************************************************
/// Constructor.
// ****************************************************************************

MainWindow::MainWindow(wxWindow *parent, wxWindowID id, const wxString &title) :
  wxFrame(parent, id, title)
{
  // ****************************************************************
  // Init the variables BEFORE the child widgets.
  // ****************************************************************

  data = Data::getInstance();

  // ****************************************************************
  // Do some unit testing.
  // ****************************************************************

  // ...

  // ****************************************************************
  /// Init the attributes and widgets of this window.
  // ****************************************************************

  // Make the background color the same as that for buttons.
  this->SetBackgroundColour(wxSystemSettings::GetColour(wxSYS_COLOUR_BTNFACE));

  initWidgets();

  // Make the main window double buffered to avoid any flickering
  // of the child-windows and during resizing.
  this->SetDoubleBuffered(true);
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::initWidgets()
{
  // ****************************************************************
  // Accellerator keys.
  // ****************************************************************

  const int NUM_ACCELS = 2;
  wxAcceleratorEntry entries[NUM_ACCELS];
  entries[0].Set(wxACCEL_CTRL, WXK_LEFT,  IDK_CTRL_LEFT);
  entries[1].Set(wxACCEL_CTRL, WXK_RIGHT, IDK_CTRL_RIGHT);

  wxAcceleratorTable accel(NUM_ACCELS, entries);
  this->SetAcceleratorTable(accel);

  // ****************************************************************
  // Set properties of this window.
  // ****************************************************************

  this->SetSize(0, 0, 800, 600);

  // ****************************************************************
  // Create the menu.
  // ****************************************************************

  wxMenu *menu = NULL;
  
  menuBar = new wxMenuBar();
  menu = new wxMenu();
  menu->Append(IDM_LOAD_AVI_FILM, "Load AVI film");
  menu->Append(IDM_LOAD_RAW_FILM, "Load (raw data) film");
  menu->Append(IDM_LOAD_BMP_SEQUENCE, "Load BMP sequence as film");
  menu->AppendSeparator();
  menu->Append(IDM_LOAD_SEGMENTATION_DATA, "Load segmentation data");
  menu->Append(IDM_SAVE_SEGMENTATION_DATA, "Save segmentation data");
  menu->Append(IDM_EXPORT_GLOTTIS_AREA, "Export glottis area waveform");
  menu->Append(IDM_EXPORT_GLOTTIS_CONTOUR, "Export glottis contour waveform");
  menu->AppendSeparator();
  menu->Append(wxID_EXIT, "Exit");

  menuBar->Append(menu, "File");

  // ****************************************************************

  menu = new wxMenu();
  menu->Append(wxID_ABOUT, "About");

  menuBar->Append(menu, "Help");

  // ****************************************************************

  this->SetMenuBar(menuBar);

  // ****************************************************************
  // Create the child widgets.
  // ****************************************************************

  wxButton *button = NULL;
  wxStaticText *label = NULL;
  wxBoxSizer *sizer = NULL;     // Help sizer

  // ****************************************************************
  // Top level sizer.
  // ****************************************************************

  wxBoxSizer *topLevelSizer = new wxBoxSizer(wxHORIZONTAL);

  // ****************************************************************
  // Left side: A separate panels is used for the left side with its
  // own sizer.
  // ****************************************************************

  wxBoxSizer *leftSizer = new wxBoxSizer(wxVERTICAL);
  wxPanel *leftPanel = new wxPanel(this);
  leftSizer->Add(leftPanel);
  topLevelSizer->Add(leftSizer, 0, wxALL, 5);

  leftSizer = new wxBoxSizer(wxVERTICAL);
  leftPanel->SetSizer(leftSizer);

  // A little space at the top.
  leftSizer->AddSpacer(5);

  // Radio buttons for the image type

  const wxString IMAGE_TYPE_CHOICES[Data::NUM_IMAGE_TYPES] = 
  { 
    "Color", 
    "Grayscale",
    "Edge picture",
    "Aligned gradient"
  };
  
  radImageType = new wxRadioBox(leftPanel, IDR_IMAGE_TYPE_BOX, "Image type", 
    wxDefaultPosition, wxDefaultSize,
    Data::NUM_IMAGE_TYPES, IMAGE_TYPE_CHOICES, Data::NUM_IMAGE_TYPES, wxRA_SPECIFY_ROWS);
  leftSizer->Add(radImageType, 0, wxALL | wxGROW, 3);

  // ****************************************************************

  chkSeedThresholdCoupling = new wxCheckBox(leftPanel, IDC_SEED_THRESHOLD_COUPLING, "Seed-threshold coupling");
  leftSizer->Add(chkSeedThresholdCoupling, 0, wxALL | wxGROW, 3);

  chkShowGlottisArea = new wxCheckBox(leftPanel, IDC_SHOW_GLOTTIS_AREA, "Show glottis area");
  leftSizer->Add(chkShowGlottisArea, 0, wxALL | wxGROW, 3);

  chkShowGlottisEdge = new wxCheckBox(leftPanel, IDC_SHOW_GLOTTIS_EDGE, "Show glottis edge");
  leftSizer->Add(chkShowGlottisEdge, 0, wxALL | wxGROW, 3);

  leftSizer->AddSpacer(5);

  leftSizer->Add(new wxButton(leftPanel, IDB_CLEAR_SEGMENTATION_DATA, 
    "Clear segmentation data"), 0, wxALL | wxGROW, 3);

  leftSizer->Add(new wxButton(leftPanel, IDB_CALC_GLOTTAL_AREA_CURVE, 
    "Calc. glottal area curve"), 0, wxALL | wxGROW, 3);

  // ****************************************************************
  // Static line to separate the left and right part.
  // ****************************************************************

  wxStaticLine *verticalLine = new wxStaticLine(this, wxID_ANY, wxDefaultPosition, 
    wxDefaultSize, wxLI_VERTICAL);

  topLevelSizer->Add(verticalLine, 0, wxGROW | wxALL, 2);

  // ****************************************************************
  // The right part of the main window.
  // ****************************************************************

  wxBoxSizer *rightSizer = new wxBoxSizer(wxVERTICAL);
  
  // Upper part with the 3 pictures

  wxBoxSizer *topSizer = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *topRightSizer = new wxBoxSizer(wxVERTICAL);

  bigPicture = new FilmPicture(this, 0);
  bigPicture->SetSize(wxSize(512, 512));
  topSizer->Add(bigPicture, 3, wxALL | wxSHAPED, 0);
  topSizer->AddSpacer(1);

  smallPicture1 = new FilmPicture(this, 1);
  smallPicture1->SetSize(wxSize(256, 256));
  topSizer->Add(smallPicture1, 2, wxALL | wxSHAPED, 0);
  topSizer->AddSpacer(1);

  smallPicture2 = new FilmPicture(this, 2);
  smallPicture2->SetSize(wxSize(256, 256));
  topSizer->Add(smallPicture2, 2, wxALL | wxSHAPED, 0);

  rightSizer->Add(topSizer, 2, wxALL | wxGROW, 0);

  // The scroll bar

  scrFrameIndex = new wxScrollBar(this, IDS_FRAME_INDEX);
  scrFrameIndex->SetMinSize(wxSize(512+256+1, 20));
  scrFrameIndex->SetScrollbar(0, 1, 8000, 1);
  rightSizer->Add(scrFrameIndex, 0, wxALL | wxGROW, 1);

  // The middle sizer

  wxBoxSizer *middleSizer = new wxBoxSizer(wxHORIZONTAL);

  button = new wxButton(this, IDB_TO_START, "To start");
  middleSizer->Add(button, 0, wxALL, 3);

  button = new wxButton(this, IDB_PLAY_ALL, "Play");
  middleSizer->Add(button, 0, wxALL, 3);

  button = new wxButton(this, IDB_PLAY_SELECTION, "Play selection");
  button->Enable(false);
  middleSizer->Add(button, 0, wxALL, 3);

  // Playback speed

  middleSizer->Add(new wxStaticText(this, wxID_ANY, "Playback speed:"), 0, wxALL, 6);

  ctrlPlaybackSpeed = new wxSpinCtrl(this, IDC_PLAYBACK_SPEED, "", wxDefaultPosition,
    wxSize(50, 20), wxSP_ARROW_KEYS, 1, 100, 1);
  middleSizer->Add(ctrlPlaybackSpeed, 0, wxALL, 3);

  rightSizer->Add(middleSizer, 0, wxGROW | wxALL, 2);

  // The bottom sizer

  wxBoxSizer *bottomSizer = new wxBoxSizer(wxHORIZONTAL);

  signalPicture = new SignalPicture(this);
  bottomSizer->Add(signalPicture, 1, wxALL | wxGROW, 0);

  scrSignals = new wxScrollBar(this, IDS_SIGNALS, wxDefaultPosition, wxDefaultSize, wxSB_VERTICAL);
  scrSignals->SetScrollbar(0, 1, 100, 1);
  scrSignals->SetMinSize(wxSize(20, -1));
  scrSignals->Enable(false);    // For now, we don't really need it.
  bottomSizer->Add(scrSignals, 0, wxALL | wxGROW, 1);
    
  rightSizer->Add(bottomSizer, 1, wxGROW | wxALL, 0);

  topLevelSizer->Add(rightSizer, 1, wxGROW);

  // ****************************************************************
  // Set the top-level sizer for this window.
  // ****************************************************************

  this->SetSizer(topLevelSizer);
  this->Fit();

  // ****************************************************************
  // Update all widgets.
  // ****************************************************************

  updateWidgets();
}

// ****************************************************************************
/// Update all widgets.
// ****************************************************************************

void MainWindow::updateWidgets()
{
  chkSeedThresholdCoupling->SetValue(data->seedThresholdCoupling);
  chkShowGlottisArea->SetValue(data->showGlottisArea);
  chkShowGlottisEdge->SetValue(data->showGlottisEdge);

  // ****************************************************************

  int currentFrame = data->getCurrFrameIndex();

  // Scrollbar for the current frame

  int numFrames = data->film->getNumFrames();
  if (currentFrame != -1)
  {
    scrFrameIndex->Enable(true);
    scrFrameIndex->SetScrollbar(currentFrame, 1, numFrames, 1);
  }
  else
  {
    scrFrameIndex->Enable(false);
  }

  // Playback speed
  ctrlPlaybackSpeed->SetValue(data->playbackSpeed_percent);

  // ****************************************************************
  // Update the pictures.
  // ****************************************************************

  bigPicture->Refresh();
  smallPicture1->Refresh();
  smallPicture2->Refresh();
  signalPicture->Refresh();
}

// ****************************************************************************
/// Update the scrollbar for the frame index.
// ****************************************************************************

void MainWindow::updateScrollbar()
{
  int numFrames = data->film->getNumFrames();
  if (numFrames > 0)
  {
    scrFrameIndex->Enable(true);

    if (data->frameIndex[0] < 0)
    {
      data->frameIndex[0] = 0;
    }
    if (data->frameIndex[0] >= numFrames)
    {
      data->frameIndex[0] = numFrames - 1;
    }
    scrFrameIndex->SetScrollbar(data->frameIndex[0], 1, numFrames, 1);
  }
  else
  {
    scrFrameIndex->Enable(false);
  }
}

// ****************************************************************************
// ****************************************************************************

void MainWindow::OnCloseWindow(wxCloseEvent &event)
{
  if (wxMessageBox("Do you really want to quit?", "Quit", wxYES_NO, this) == wxYES)
  {
    this->Destroy();
    exit(0);
  }
}


// ****************************************************************************
/// Load an AVI film.
// ****************************************************************************

void MainWindow::OnLoadAviFilm(wxCommandEvent &event)
{
  wxFileName fileName(data->filmFileName);

  wxString name = wxFileSelector("Open a film", (wxChar*)0,
    fileName.GetFullName(), ".avi", "Film files (*.avi)|*.avi", 
    wxOPEN | wxFILE_MUST_EXIST, this);

  if (name.empty() == false)
  {
    data->filmFileName = name;
    if (data->film->loadAviFilm(data->filmFileName))
    {
      data->resetSrgData();
      updateWidgets();
      this->SetTitle("GlottalImageExplorer: " + data->filmFileName);
    }
    else
    {
      wxMessageBox("Loading AVI film failed.", "Error");
      this->SetTitle("GlottalImageExplorer");
    }

    for (int i = 0; i < Data::NUM_DISPLAYED_FRAMES; i++)
    {
      data->frameIndex[i] = 0;
    }
  }
}

// ****************************************************************************
/// Load a (raw data) film.
// ****************************************************************************

void MainWindow::OnLoadRawFilm(wxCommandEvent &event)
{
  wxFileName fileName(data->filmFileName);

  wxString name = wxFileSelector("Open a film", (wxChar*)0,
    fileName.GetFullName(), ".bld", "Film files (*.bld)|*.bld",
    wxOPEN | wxFILE_MUST_EXIST, this);

  if (name.empty() == false)
  {
    data->filmFileName = name;
    if (data->film->loadRawFilm(data->filmFileName))
    {
      data->resetSrgData();
      updateWidgets();
      this->SetTitle("GlottalImageExplorer: " + data->filmFileName);
    }
    else
    {
      wxMessageBox("Loading raw data film failed.", "Error");
      this->SetTitle("GlottalImageExplorer");
    }

    for (int i = 0; i < Data::NUM_DISPLAYED_FRAMES; i++)
    {
      data->frameIndex[i] = 0;
    }
  }
}


// ****************************************************************************
/// Load a bmp image sequence as film.
// ****************************************************************************

void MainWindow::OnLoadBmpSequence(wxCommandEvent &event)
{
  wxDirDialog dialog(this, "Please select a folder with bmp files", data->bmpFolderName);
  if (dialog.ShowModal() == wxID_OK)
  {
    wxString path = dialog.GetPath();
    data->bmpFolderName = path;
    
    if (data->film->loadBmpSequence(path))
    {
      data->resetSrgData();
      updateWidgets();
      this->SetTitle("GlottalImageExplorer: BMP folder " + data->bmpFolderName);
    }
    else
    {
      wxMessageBox("Loading failed.", "Error");
      this->SetTitle("GlottalImageExplorer");
    }

    for (int i = 0; i < Data::NUM_DISPLAYED_FRAMES; i++)
    {
      data->frameIndex[i] = 0;
    }
  }
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnLoadSegmentationData(wxCommandEvent &event)
{
  if (data->film->getNumFrames() < 1)
  {
    wxMessageBox("No film loaded.", "Error!");
    return;
  }

  wxString name = wxFileSelector("Load segmentation data", (wxChar*)0,
    wxFileName(data->segmentationFileName).GetFullName(), "seg", "seg-files (*.seg)|*.seg",
    wxFD_OPEN | wxFD_FILE_MUST_EXIST, this);

  if (name.empty() == false)
  {
    data->segmentationFileName = name;
    if (!data->loadSegmentationData(name.c_str()))
    {
      wxMessageBox("Error in loading the file.", "Error!");
    }
    updateWidgets();
  }
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnSaveSegmentationData(wxCommandEvent &event)
{
  if (data->film->getNumFrames() < 1)
  {
    wxMessageBox("No film loaded.", "Error!");
    return;
  }

  wxString name = wxFileSelector("Save segmentation data", (wxChar*)0,
    wxFileName(data->segmentationFileName).GetFullName(), "seg", "seg-files (*.seg)|*.seg",
    wxFD_SAVE | wxFD_OVERWRITE_PROMPT, this);

  if (name.empty() == false)
  {
    data->segmentationFileName = name;

    if (!data->saveSegmentationData(name.c_str()))
    {
      wxMessageBox("Error saving the file.", "Error!");
    }
  }
}

// ****************************************************************************
// ****************************************************************************

void MainWindow::OnExportGlottisArea(wxCommandEvent &event)
{
  if (data->film->getNumFrames() < 1)
  {
    wxMessageBox("No film loaded.", "Error!");
    return;
  }

  wxString name = wxFileSelector("Export glottis areas", (wxChar*)0,
    wxFileName(data->areaFileName).GetFullName(), "txt", "txt-files (*.txt)|*.txt",
    wxFD_SAVE | wxFD_OVERWRITE_PROMPT, this);

  if (name.empty() == false)
  {
    data->areaFileName = name;

    if (!data->exportGlottisArea(name.c_str()))
    {
      wxMessageBox("Error saving the file.", "Error!");
    }
  }
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnExportGlottisContour(wxCommandEvent &event)
{
  if (data->film->getNumFrames() < 1)
  {
    wxMessageBox("No film loaded.", "Error!");
    return;
  }

  wxString name = wxFileSelector("Export glottis contours", (wxChar*)0,
    wxFileName(data->contourFileName).GetFullName(), "txt", "txt-files (*.txt)|*.txt",
    wxFD_SAVE | wxFD_OVERWRITE_PROMPT, this);

  if (name.empty() == false)
  {
    data->contourFileName = name;

    if (!data->exportGlottisContour(name.c_str()))
    {
      wxMessageBox("Error saving the file.", "Error!");
    }
  }
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnExit(wxCommandEvent &event)
{
  Close(true);
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnAbout(wxCommandEvent &event)
{
  wxMessageDialog dialog(this, 
    "This is GlottalImageExplorer v1.0. Copyright by Peter Birkholz, 2016.");
  dialog.ShowModal();
}

// ****************************************************************************
// ****************************************************************************

void MainWindow::OnSelectImageType(wxCommandEvent &event)
{
  int selection = event.GetSelection();
  data->imageType = selection;
  updateWidgets();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnSeedThresholdCoupling(wxCommandEvent &event)
{
  if (data->seedThresholdCoupling)
  {
    data->seedThresholdCoupling = false;
  }
  else
  {
    if (wxMessageBox("Dou you really want to couple the seed positions with the "
      "threshold points? This might change the current threshold gradients.",
      "Are you sure?", wxYES_NO) == wxYES)
    {
      data->seedThresholdCoupling = true;

      // Adapt all the threshold y-values to the seeds!

      int i, k;
      int numFrames = data->film->getNumFrames();

      for (i = 0; i < numFrames; i++)
      {
        for (k = 0; k < Data::NUM_SEEDS; k++)
        {
          data->srgData[i].thresholdY[k] = data->srgData[i].seed[k].y;
        }
      }

    }
  }

  updateWidgets();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnShowGlottisArea(wxCommandEvent &event)
{
  data->showGlottisArea = !data->showGlottisArea;
  updateWidgets();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnShowGlottisEdge(wxCommandEvent &event)
{
  data->showGlottisEdge = !data->showGlottisEdge;
  updateWidgets();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnClearSegmentationData(wxCommandEvent &event)
{
  if (wxMessageBox("Do you really want to clear all segmentation data?", 
    "Are you sure?", wxYES_NO) == wxYES)
  {
    data->resetSrgData();
    updateWidgets();
  }
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnCalcGlottalAreaCurve(wxCommandEvent &event)
{
  int i;
  int numFrames = data->film->getNumFrames();
  int numGlottisPixels = 0;
  int segmentMatrix[Film::WIDTH * Film::HEIGHT];
  int leftContour[Film::HEIGHT];
  int rightContour[Film::HEIGHT];

  if (numFrames < 1)
  {
    return;
  }

  int maxProgressValue = numFrames / 64;

  wxProgressDialog dialog("Calculation progress",
    "Please wait until the calculation finished", maxProgressValue, this,
    wxPD_APP_MODAL | wxPD_ELAPSED_TIME | wxPD_AUTO_HIDE);

  for (i = 0; i < numFrames; i++)
  {
    data->srgData[i].glottisArea_pix = ImageProc::segmentGlottis(&data->film->getFrame(i), 
      leftContour, rightContour, segmentMatrix, &data->srgData[i]);

    if ((i & 63) == 0)
    {
      dialog.Update(i / 64);
    }
  }
  dialog.Update(maxProgressValue);

  updateWidgets();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnPlaybackThreadEvent(wxCommandEvent& event)
{
  int n = event.GetInt();

  // The thread reached its end - either normally or by the call
  // to wxThread::Destroy()

  if (n == -1)
  {
    progressDialog->Destroy();
    progressDialog = NULL;

    // The dialog is aborted because the event came from another thread, so
    // we may need to wake up the main event loop for the dialog to be
    // really closed
    wxWakeUpIdle();

    // Refresh all pictures and controls.
    updateWidgets();
  }
  else

  // The event asks us to update the progress dialog state
  {
    if (progressDialog->Update(n) == false)
    {
      // If the update of the dialog failed, the user pressed the cancel button.
      // Therefore, tell the synthesis thread to cancel.
      // In respons, the thread will soon send the final event with the id -1.

      playbackThread->cancelNow();
    }

    updateScrollbar();

    // Repaint the pictures IMMEDIATELY.
    bigPicture->Refresh();
    bigPicture->Update();
    playbackThread->signalGuiUpdateFinished();
  }

}


// ****************************************************************************
/// Move the frame index to the first frame.
// ****************************************************************************

void MainWindow::OnToStart(wxCommandEvent &event)
{
  data->frameIndex[0] = 0;
  updateWidgets();
}


// ****************************************************************************
/// Playback the whole film.
// ****************************************************************************

void MainWindow::OnStartPlayback(wxCommandEvent &event)
{
  if ((data->frameIndex < 0) || (data->frameIndex[0] >= data->film->getNumFrames()))
  {
    return;
  }

  // ****************************************************************
  // Create the playback thread (it will be destroyed automatically
  // when it ends).
  // ****************************************************************

  playbackThread = new PlaybackThread(this);
  if (playbackThread->Create() != wxTHREAD_NO_ERROR)
  {
    printf("ERROR: Can't create playback thread!");
    return;
  }
  playbackThread->setFrameRange(data->frameIndex[0], data->film->getNumFrames() - 1);

  // Create the progress dialog

  progressDialog = new wxProgressDialog("Playback progress",
    "Wait until the playback finished or press [Cancel]", 100, this,
    wxPD_CAN_ABORT | wxPD_APP_MODAL | wxPD_ELAPSED_TIME);
    
  // Start the playback thread.
  // It will now continually produce events during processing and when it ends.
  playbackThread->Run();
}

// ****************************************************************************
/// Playback the selected sequence.
// ****************************************************************************

void MainWindow::OnStartSelectionPlayback(wxCommandEvent &event)
{
  if ((data->frameIndex < 0) || (data->frameIndex[0] >= data->film->getNumFrames()))
  {
    return;
  }

  // ****************************************************************
  // Create the playback thread (it will be destroyed automatically
  // when it ends).
  // ****************************************************************

  playbackThread = new PlaybackThread(this);
  if (playbackThread->Create() != wxTHREAD_NO_ERROR)
  {
    printf("ERROR: Can't create playback thread!");
    return;
  }
  playbackThread->setFrameRange(data->selectionIndex[0], data->selectionIndex[1]);

  // Create the progress dialog

  progressDialog = new wxProgressDialog("Playback progress",
    "Wait until the playback finished or press [Cancel]", 100, this,
    wxPD_CAN_ABORT | wxPD_APP_MODAL | wxPD_ELAPSED_TIME);
    
  // Start the playback thread.
  // It will now continually produce events during processing and when it ends.
  playbackThread->Run();
}

// ****************************************************************************
// ****************************************************************************

void MainWindow::OnPlaybackSpeedChanged(wxSpinEvent &event)
{
  data->playbackSpeed_percent = ctrlPlaybackSpeed->GetValue();
}

// ****************************************************************************
// ****************************************************************************

void MainWindow::OnFrameIndexScroll(wxScrollEvent &event)
{
  data->frameIndex[0] = scrFrameIndex->GetThumbPosition();
  updateWidgets();
}

// ****************************************************************************
// ****************************************************************************

void MainWindow::OnSignalPictureScroll(wxScrollEvent &event)
{

  updateWidgets();
}


// ****************************************************************************
// Go one frame left.
// ****************************************************************************

void MainWindow::OnKeyCtrlLeft(wxCommandEvent &event)
{
  if (data->frameIndex[0] > 0)
  {
    data->frameIndex[0]--;
  }
  updateWidgets();
}

// ****************************************************************************
// Go one frame right.
// ****************************************************************************

void MainWindow::OnKeyCtrlRight(wxCommandEvent &event)
{
  if (data->frameIndex[0] < data->film->getNumFrames() - 1)
  {
    data->frameIndex[0]++;
  }
  updateWidgets();
}

// ****************************************************************************

